//== INCLUDES =================================================================

#include "performance.h"
#include <Mesquite_all_headers.hpp>
#include <set>
#include <algorithm>

using namespace Mesquite;


//== CLASS DEFINITION =========================================================


class MyMesh : public MeshImpl
{
public:

    MyMesh() : MeshImpl() {}


    unsigned int n_vertices() const { return myMesh->num_vertices(); }
    unsigned int n_faces()    const { return myMesh->num_elements(); }


    bool read_off(const char* _filename)
    {
        MsqError err;

        std::ifstream ifs(_filename);
        if (!ifs) return false;

        // header
        std::string s;
        ifs >> s;
        if (s != "OFF") return false;
        uint nv, ne, nf;
        ifs >> nv >> nf >> ne;

        // vertices
        Vector3D p;
        for (uint i=0; i<nv; ++i)
        {
            ifs >> p[0] >> p[1] >> p[2];
            myMesh->add_vertex(p, false, err);
        }

        // faces
        std::vector<size_t> indices;
        int n;
        for (uint i=0; i<nf; ++i)
        {
            ifs >> n;
            indices.resize(n);
            for (int j=0; j<n; ++j)
                ifs >> indices[j];

            switch (n)
            {
                case 3:
                    myMesh->add_element(indices, TRIANGLE, err);
                    break;

                case 4:
                    myMesh->add_element(indices, QUADRILATERAL, err);
                    break;

                default:
                    myMesh->add_element(indices, POLYGON, err);
                    break;
            }

        }

        ifs.close();
        return true;
    }


    bool write_off(const char* _filename)
    {
        MsqError err;

        std::ofstream ofs(_filename);
        if (!ofs) return false;

        // header
        ofs << "OFF\n";
        ofs << n_vertices() << ' ' << n_faces() << " 0\n";

        // vertices
        VertexIterator* vit=vertex_iterator(err);
        MsqVertex p;
        do
        {
            Mesh::VertexHandle vh = **vit;
            vertices_get_coordinates(&vh, &p, 1, err);
            ofs << p << std::endl;
            ++(*vit);
        }
        while (!vit->is_at_end());
        delete vit;

        // faces
        ElementIterator* fit=element_iterator(err);
        std::vector<Mesh::VertexHandle>  vertices;
        std::vector<size_t>              offsets;
        Mesh::ElementHandle              eh;
        do
        {
            eh = **fit;
            elements_get_attached_vertices(&eh, 1, vertices, offsets, err);
            uint n = vertices.size();
            ofs << n << ' ';
            for (uint i=0; i<n; ++i)
                ofs << (size_t)vertices[i] << ' ';
            ofs << std::endl;
            ++(*fit);
        }
        while (!fit->is_at_end());
        delete fit;

        ofs.close();
        return true;
    }



    void get_adjacent_vertices(Mesh::VertexHandle vh, std::vector<Mesh::VertexHandle>& one_ring)
    {
        static std::vector<Mesh::VertexHandle>  vertices;
        static std::vector<Mesh::ElementHandle> elements;
        static std::vector<size_t>              offsets;
        static std::set<Mesh::VertexHandle>     adj;
        Mesquite::MsqError err;

        adj.clear();
        one_ring.clear();
        elements.clear();
        vertices.clear();
        offsets.clear();

        vertices_get_attached_elements(&vh, 1, elements, offsets, err);
        elements_get_attached_vertices(&elements[0], elements.size(), vertices, offsets, err);

        for (unsigned int e=0; e<elements.size(); ++e)
        {
            const unsigned int offset = offsets[e];
            const unsigned int nv = offsets[e+1]-offset;

            for (unsigned int i=0; i<nv; ++i)
            {
                if (vertices[i+offset] == vh)
                {
                    adj.insert( vertices[ offset+((i+1)%nv) ] );
                    adj.insert( vertices[ offset+((i+nv-1)%nv) ] );
                }
            }
        }

        std::copy(adj.begin(), adj.end(), std::back_inserter(one_ring));
    }

};


//== CLASS DEFINITION =========================================================


class Mesquite_performance : public Performance_test
{

public:

    Mesquite_performance() : Performance_test()
    {
        fnormals = mesh.tag_create("fnormals", Mesquite::Mesh::DOUBLE, 3, NULL, err);
        vnormals = mesh.tag_create("vnormals", Mesquite::Mesh::DOUBLE, 3, NULL, err);
    }

    ~Mesquite_performance()
    {
        mesh.tag_delete(fnormals, err);
        mesh.tag_delete(vnormals, err);
    }

private:

    MyMesh mesh;
    TagHandle fnormals, vnormals;
    Mesquite::MsqError err;


private:

    virtual bool read_mesh(const char* _filename)
    {
        return mesh.read_off(_filename);
    }


    virtual bool write_mesh(const char* _filename)
    {
        return mesh.write_off(_filename);
    }


    virtual int circulator_test()
    {
        std::vector<Mesh::VertexHandle>  vertices;
        std::vector<Mesh::ElementHandle> elements;
        std::vector<size_t>              offsets;
        Mesh::VertexHandle               vh;
        Mesh::ElementHandle              eh;
        int                              c = 0;


        VertexIterator* vit=mesh.vertex_iterator(err);
        do
        {
            vh = **vit;
            mesh.vertices_get_attached_elements(&vh, 1, elements, offsets, err);
            c += elements.size();
            ++(*vit);
        }
        while (!vit->is_at_end());
        delete vit;


        ElementIterator* fit=mesh.element_iterator(err);
        do
        {
            eh = **fit;
            mesh.elements_get_attached_vertices(&eh, 1, vertices, offsets, err);
            c -= vertices.size();
            ++(*fit);
        }
        while (!fit->is_at_end());
        delete fit;

        return c;
    }


    virtual void barycenter_test()
    {
        VertexIterator* vit = mesh.vertex_iterator(err);

        Mesh::VertexHandle vh;
        MsqVertex    v;
        MsqVertex    p(0,0,0);


        vit->restart();
        do
        {
            vh = **vit;
            mesh.vertices_get_coordinates(&vh, &v, 1, err);
            p += v;
            ++(*vit);
        }
        while (!vit->is_at_end());

        p /= (float)mesh.n_vertices();

        vit->restart();
        do
        {
            vh = **vit;
            mesh.vertices_get_coordinates(&vh, &v, 1, err);
            v -= p;
            mesh.vertex_set_coordinates(vh, v, err);
            ++(*vit);
        }
        while (!vit->is_at_end());

        delete vit;
    }


    virtual void normal_test()
    {
        ElementIterator* fit = mesh.element_iterator(err);
        VertexIterator*  vit = mesh.vertex_iterator(err);

        std::vector<Mesh::VertexHandle>  vertices;
        std::vector<Mesh::ElementHandle> elements;
        std::vector<size_t>              offsets;
        std::vector<MsqVertex>           coords;
        Mesh::VertexHandle               vh;
        Mesh::ElementHandle              eh;;
        Vector3D                         n;


        fit->restart();
        do
        {
            eh = **fit;
            mesh.elements_get_attached_vertices(&eh, 1, vertices, offsets, err);

            coords.resize(vertices.size());
            mesh.vertices_get_coordinates(&vertices[0], &coords[0], vertices.size(), err);

            n = ((coords[0]-coords[1]) * (coords[2]-coords[1]));
            n.normalize();

            mesh.tag_set_element_data(fnormals, 1, &eh, &n, err);

            ++(*fit);
        }
        while (!fit->is_at_end());


        vit->restart();
        do
        {
            vh = **vit;
            mesh.vertices_get_attached_elements(&vh, 1, elements, offsets, err);

            Vector3D nn(0,0,0);
            for (uint j=0; j<elements.size(); ++j)
            {
                mesh.tag_get_element_data(fnormals, 1, &elements[j], &n, err);
                nn += n;
            }
            nn.normalize();

            mesh.tag_set_vertex_data(vnormals, 1, &vh, &nn, err);

            ++(*vit);
        }
        while (!vit->is_at_end());


        delete fit;
        delete vit;
    }



    virtual void smoothing_test()
    {
        VertexIterator*  vit = mesh.vertex_iterator(err);

        std::vector<Mesh::VertexHandle>  vertices;
        std::vector<Mesh::ElementHandle> elements;
        std::vector<size_t>              offsets;
        std::vector<MsqVertex>           coords;
        Mesh::VertexHandle               vh;


        vit->restart();
        do
        {
            vh = **vit;

//            mesh.vertices_get_attached_elements(&vh, 1, elements, offsets, err);
//            mesh.elements_get_attached_vertices(&elements[0], elements.size(), vertices, offsets, err);
            mesh.get_adjacent_vertices(vh, vertices);

            coords.resize(vertices.size());
            mesh.vertices_get_coordinates(&vertices[0], &coords[0], vertices.size(), err);

            Vector3D p(0,0,0);
            uint c(0);
            for (uint i=0; i<vertices.size(); ++i)
            {
                if (vertices[i] != vh)
                {
                    ++c;
                    p += coords[i];
                }
            }
            p /= (float) c;

            mesh.vertex_set_coordinates(vh, p, err);

            ++(*vit);
        }
        while (!vit->is_at_end());

        delete vit;
    }

};


//=============================================================================
